<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Product extends Model
{

    protected $fillable = [
        'product_name',
        'stock',
        'unit_id',
    ];


    public function unit()
    {
        return $this->belongsTo(Unit::class,'unit_id','id');
    }

    public function supplier()
    {
        return $this->belongsTo(Supplier::class,'supplier_id','id');
    }

    /**
     * Get the base/smallest unit of this product.
     * If it has a child conversion, use that as base unit and adjust buying price.
     * Otherwise, return the product's own unit.
     */
    public function baseUnit()
    {
        // Load child conversion (smaller unit)
        $childConversion = $this->unit->childConversions()->with('childUnit')->first();

        if ($childConversion && $childConversion->childUnit) {
            $baseUnit = $childConversion->childUnit;

            // Adjust buying price per base unit
            $baseUnit->buying_price = $this->buying_price / $childConversion->multiplier;

            return $baseUnit;
        }

        // No smaller unit, use current unit
        $unit = $this->unit;
        $unit->buying_price = $this->buying_price;

        return $unit;
    }
    public function toBaseUnits(float $qty): float
    {
        $unit = $this->unit;

        // If unit itself is base → no conversion
        if ($unit->is_base) {
            return $qty;
        }

        // Get conversion (e.g Bail → Packet)
        $conversion = $unit->conversions()->first();

        if (!$conversion) {
            throw new \Exception("No unit conversion defined for {$unit->name}");
        }

        return $qty * $conversion->multiplier;
    }
    public function purchases()
    {
        return $this->hasMany(Purchases::class, 'product_id', 'id');
    }



    public function foodIngredients()
    {
        return $this->hasMany(FoodIngredient::class);
    }
    // Ingredients used in orders
    public function orderIngredients()
    {
        return $this->hasMany(Kitchen_order_ingredient::class);
    }
}
