<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Kitchen_order_ingredient;
use App\Models\Product;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function showIngredientsReportForm() {
        return view('backend.reports.ingredients_report'); // form to select date
    }

    //
    public function ingredientReport(Request $request)
    {
        // Step 0: Validate date (optional)
        $date = $request->date ?? now()->toDateString();

        // Step 1: Load all products
        $products = Product::all();

        $report = [];

        foreach ($products as $product) {

            // Current stock
            $currentStock = $product->stock;

            // Purchased today
            $purchasedQty = $product->purchases()
                ->whereDate('purchase_date', $date)
                ->sum('product_qty');

            // Used in approved kitchen orders on that day
            $usedQty = Kitchen_order_ingredient::whereHas('order', function($q) use ($date) {
                $q->where('status', 'approved')
                    ->whereDate('order_date', $date);
            })
                ->where('product_id', $product->id)
                ->sum('required_qty');

            // Remaining calculation
            $remaining = $currentStock + $purchasedQty - $usedQty;

            $report[] = (object)[
                'product_name' => $product->product_name,
                'current_stock' => $currentStock,
                'purchased_qty' => $purchasedQty,
                'used_qty' => $usedQty,
                'remaining' => $remaining,
            ];
        }

        return view('backend.reports.ingredients_report', compact('report', 'date'));
    }

    protected function generateIngredientReportData($date)
    {
        $products = Product::all();
        $report = [];

        foreach ($products as $product) {
            $currentStock = $product->stock;
            $purchasedQty = $product->purchases()
                ->whereDate('purchase_date', $date)
                ->sum('product_qty');
            $usedQty = Kitchen_order_ingredient::whereHas('order', function($q) use ($date) {
                $q->where('status', 'approved')
                    ->whereDate('order_date', $date);
            })
                ->where('product_id', $product->id)
                ->sum('required_qty');
            $remaining = $currentStock + $purchasedQty - $usedQty;

            $report[] = (object)[
                'product_name' => $product->product_name,
                'current_stock' => $currentStock,
                'purchased_qty' => $purchasedQty,
                'used_qty' => $usedQty,
                'remaining' => $remaining,
            ];
        }

        return $report;
    }
    public function ingredientReportPdf(Request $request)
    {
        $date = $request->date ?? now()->toDateString();

        $report = $this->generateIngredientReportData($date); // same logic as above

        $pdf = PDF::loadView('backend.reports.ingredients_report_pdf', compact('report', 'date'));

        return $pdf->download("ingredients_report_{$date}.pdf");
    }
    // Show form to select date
    public function showRecipesReportForm()
    {
        return view('backend.reports.recipes_report'); // blade with date picker
    }

// Generate the HTML report
    public function recipesReport(Request $request)
    {
        $date = $request->date ?? now()->toDateString();

        // Approved quantities
        $approved = \App\Models\Kitchen_order_approval::whereDate('order_date', $date)
            ->selectRaw('recipe_id, recipe_name, SUM(approved_qty) as approved_qty')
            ->groupBy('recipe_id', 'recipe_name')
            ->get()
            ->keyBy('recipe_id');

        // Sold quantities
        $sold = \App\Models\Sale_item::whereHas('sale', function($q) use ($date) {
            $q->whereDate('created_at', $date);
        })
            ->selectRaw('recipe_id, SUM(qty) as sold_qty')
            ->groupBy('recipe_id')
            ->get()
            ->keyBy('recipe_id');

        $report = [];
        foreach ($approved as $recipeId => $data) {
            $soldQty = $sold[$recipeId]->sold_qty ?? 0;
            $report[] = (object)[
                'recipe_name'   => $data->recipe_name,
                'approved_qty'  => $data->approved_qty,
                'sold_qty'      => $soldQty,
                'remaining_qty' => $data->approved_qty - $soldQty,
            ];
        }

        return view('backend.reports.recipes_report', compact('report', 'date'));
    }

// PDF export
    public function recipesReportPdf(Request $request)
    {
        $date = $request->date ?? now()->toDateString();

        // Reuse the same data generation
        $approved = \App\Models\Kitchen_order_approval::whereDate('order_date', $date)
            ->selectRaw('recipe_id, recipe_name, SUM(approved_qty) as approved_qty')
            ->groupBy('recipe_id', 'recipe_name')
            ->get()
            ->keyBy('recipe_id');

        $sold = \App\Models\Sale_item::whereHas('sale', function($q) use ($date) {
            $q->whereDate('created_at', $date);
        })
            ->selectRaw('recipe_id, SUM(qty) as sold_qty')
            ->groupBy('recipe_id')
            ->get()
            ->keyBy('recipe_id');

        $report = [];
        foreach ($approved as $recipeId => $data) {
            $soldQty = $sold[$recipeId]->sold_qty ?? 0;
            $report[] = (object)[
                'recipe_name'   => $data->recipe_name,
                'approved_qty'  => $data->approved_qty,
                'sold_qty'      => $soldQty,
                'remaining_qty' => $data->approved_qty - $soldQty,
            ];
        }
        $report = collect($report);
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('backend.reports.recipes_report_pdf', compact('report', 'date'));

        return $pdf->download("recipes_report_{$date}.pdf");
    }



}
