<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Purchases;
use App\Models\Supplier;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    //
    public function AllPurchases ()
    {

        $purchases = Purchases::latest()->get();
        return view('backend.purchases.all_purchases',compact('purchases'));
    }
    public function AddPurchase(){
        $units = Unit::latest()->get();
        $supplier = Supplier::latest()->get();
        $product = Product::latest()->get();
        return view('backend.purchases.add_purchase',compact('units','supplier','product'));
    } // End Method

    public function StorePurchase(Request $request)
    {
        $request->validate([
            'product_id'    => 'required|exists:products,id',
            'supplier_id'   => 'required|exists:suppliers,id',
            'product_qty'   => 'required|numeric|min:0.01',
            'buying_price'  => 'required|numeric|min:0.01',
            'purchase_date' => 'required|date',
        ]);

        DB::transaction(function () use ($request) {

            // 1️⃣ Get the product
            $product = Product::findOrFail($request->product_id);

            // 2️⃣ Check for a child/base unit conversion
            $childConversion = $product->unit->childConversions()->first(); // check for smaller/base unit

            if ($childConversion) {
                // Product has a base unit -> multiply quantity by multiplier
                $quantityInBase = $request->product_qty * $childConversion->multiplier;
            } else {
                // No smaller unit -> store as is
                $quantityInBase = $request->product_qty;
            }

            // 3️⃣ Update stock in base units
            $product->stock += $quantityInBase;
            $product->save();

            // 4️⃣ Insert into purchases table (store original purchased quantity)
            Purchases::create([
                'product_id'   => $request->product_id,
                'supplier_id'  => $request->supplier_id,
                'unit_id'      => $product->unit_id, // original purchased unit
                'product_qty'  => $request->product_qty, // original purchased quantity
                'buying_price' => $request->buying_price,
                'purchase_date'=> $request->purchase_date,
            ]);

        });

        $notification = [
            'message' => 'Purchase Updated Successfully',
            'alert-type' => 'success'
        ];

        return redirect()->route('all.purchases')->with($notification);
    }


    public function EditPurchase($id)
    {
        $purchase = Purchases::findOrFail($id);

        // get ALL products for select dropdown
        $products = Product::all();
        $units = Unit::all();
        $supplier = Supplier::all();

        return view(
            'backend.purchases.edit_purchase',
            compact('purchase', 'products','units','supplier')
        );
    }
    public function UpdatePurchase(Request $request)
    {
        $id = $request->id;
        $request->validate([
            'product_id'    => 'required|exists:products,id',
            'supplier_id'   => 'required|exists:suppliers,id',
            'product_qty'   => 'required|numeric|min:0.01',
            'buying_price'  => 'required|numeric|min:0.01',
            'purchase_date' => 'required|date',
        ]);

        DB::transaction(function () use ($request, $id) {

            // 1️⃣ Get existing purchase
            $purchase = Purchases::findOrFail($id);

            // 2️⃣ Get product
            $product = Product::findOrFail($purchase->product_id);

            // 3️⃣ REVERSE old stock
            $product->decrement('stock', $purchase->product_qty);

            // 4️⃣ APPLY new stock
            $product->increment('stock', $request->product_qty);

            // 5️⃣ Update purchase
            $purchase->update([
                'product_id'    => $request->product_id,
                'supplier_id'   => $request->supplier_id,
                'product_qty'   => $request->product_qty,
                'buying_price'  => $request->buying_price,
                'purchase_date' => $request->purchase_date,
                'unit_id'       => $product->unit_id,
            ]);
        });
        $notification = array(
            'message' => 'Product Updated Successfully',
            'alert-type' => 'success'
        );
        return redirect()
            ->route('all.purchases')
            ->with($notification);
    }

}
