<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Kitchen_order;
use App\Models\Kitchen_order_ingredient;
use App\Models\Recipe;
use App\Models\Register;
use App\Models\Sale;
use App\Models\Sale_item;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PosController extends Controller
{
    //
    public function index()
    {
        $userId = auth()->id();

        // 1️⃣ Check if this user has an open register
        $register = Register::where('user_id', $userId)
            ->where('status', 'open')
            ->first();

        if (!$register) {
            return redirect()
                ->route('register.open')
                ->with('error', 'You need to open the register first.');
        }

        // 2️⃣ Load ONLY kitchen ingredients for this register
        $registerProducts = Kitchen_order_ingredient::where('register_id', $register->id)
            ->whereHas('order', fn($q) => $q->where('status', 'approved'))
            ->with('recipe')
            ->get();

        // 3️⃣ If there are no products, redirect to add.order
        if ($registerProducts->isEmpty()) {
            return redirect()
                ->route('add.order')
                ->with('info', 'No approved products in this register. Please create your order sheet first.');
        }

        // 4️⃣ Prepare products for Blade (no aggregation)
        $recipes = $registerProducts->map(fn($ingredient) => [
            'recipe_id'     => $ingredient->recipe_id,
            'recipe_name'   => $ingredient->recipe_name,
            'available_qty' => $ingredient->recipe_qty,
            'selling_price' => $ingredient->recipe->selling_price ?? 0,
        ]);

        // 5️⃣ Load POS view
        return view('backend.pos.pos_index', compact('register', 'recipes'));
    }







    // AJAX: Check if a product is available before adding to cart
    public function checkProductQuantity(Request $request)
    {
        $ingredient = Kitchen_order_ingredient::find($request->ingredient_id);
        if (!$ingredient) {
            return response()->json(['status' => 'error', 'message' => 'Ingredient not found']);
        }

        return response()->json([
            'status' => 'success',
            'available_qty' => $ingredient->stock_available - $ingredient->required_qty
        ]);
    }
    public function completeSale(Request $request)
    {
        try {
            // ✅ Get the open register for this user
            $register = Register::where('user_id', auth()->id())
                ->where('status', 'open')
                ->first();

            if (!$register) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No open register found'
                ], 422);
            }

            $saleItems = $request->input('saleItems');

            if (empty($saleItems)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No items to sell'
                ], 422);
            }

            DB::beginTransaction();

            $totalAmount = 0;

            // 🆕 Create sale
            $sale = $register->sales()->create([
                'user_id'     => auth()->id(),
                'total_amount'=> 0,
            ]);

            foreach ($saleItems as $item) {
                $recipe = Recipe::find($item['id']);
                if (!$recipe) {
                    throw new \Exception("Recipe not found");
                }

                $qty       = (int) $item['qty'];
                $unitPrice = (float) $recipe->selling_price;
                $total     = $qty * $unitPrice;
                $totalAmount += $total;

                // ✅ Get approved kitchen stock linked to this register
                $kitchenIngredient = Kitchen_order_ingredient::where('recipe_id', $recipe->id)
                    ->where('register_id', $register->id)
                    ->lockForUpdate()
                    ->first();

                if (!$kitchenIngredient || $kitchenIngredient->recipe_qty < $qty) {
                    throw new \Exception(
                        "Insufficient approved kitchen stock for {$recipe->recipe_name}"
                    );
                }

                // Deduct approved quantity
                $kitchenIngredient->recipe_qty -= $qty;
                $kitchenIngredient->save();

                // Save sale item
                $sale->items()->create([
                    'kitchen_order_id'            => $kitchenIngredient->kitchen_order_id,
                    'kitchen_order_ingredient_id' => $kitchenIngredient->id,
                    'recipe_id'                   => $recipe->id,
                    'recipe_name'                 => $recipe->recipe_name,
                    'qty'                         => $qty,
                    'unit_price'                  => $unitPrice,
                    'total_price'                 => $total,
                ]);
            }

            // Update total amount
            $sale->update(['total_amount' => $totalAmount]);

            DB::commit();

            return response()->json([
                'status'  => 'success',
                'message' => 'Sale completed successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }







}
