<?php

namespace App\Http\Controllers\Backend;

use App\Exports\MonthlyPayrollExport;
use App\Http\Controllers\Controller;
use App\Models\Employee;
use App\Models\Employee_loan_deduction;
use App\Models\Payroll;
use App\Models\Payroll_run;
use App\Models\Payroll_run_employee;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class PayrollController extends Controller
{
    //
    public function index()
    {
        $employees = Employee::orderBy('full_name')->get();
        return view('backend.payroll.payroll_index',compact('employees'));
    }

    public function runPayroll(Payroll_run $run)
    {
        // 1️⃣ Ensure payroll is approved
        if ($run->status !== 'approved') {
            return back()->with('error', 'Payroll must be approved first');
        }

        $payPeriod = $run->pay_period; // Use pay_period from the Payroll_run

        // 2️⃣ Get all employees assigned to this run
        $employeeIds = $run->employees->pluck('employee_id')->toArray();

        $employees = Employee::with([
            'allowances',           // employee_allowances → allowances
            'deductions',           // employee_deductions → deductions
            'loans.deductions'      // employee_loans → employee_loan_deductions
        ])->whereIn('id', $employeeIds)->get();

        foreach ($employees as $employee) {

            $basicSalary = $employee->basic_salary ?? 0;

            // 3️⃣ Calculate total allowances
            $totalAllowances = $employee->allowances->sum(function ($allowance) use ($basicSalary) {
                return $allowance->type === 'fixed'
                    ? $allowance->amount
                    : ($basicSalary * $allowance->amount / 100);
            });

            // 4️⃣ Calculate total other deductions
            $otherDeductions = $employee->deductions->sum(function ($deduction) use ($basicSalary) {
                return $deduction->type === 'fixed'
                    ? $deduction->amount
                    : ($basicSalary * $deduction->amount / 100);
            });

            // 5️⃣ Handle loan deductions
            $loanDeductions = 0;
            foreach ($employee->loans as $loan) {
                if ($loan->remaining_amount > 0) {
                    $monthlyDeduction = $loan->remaining_amount; // Deduct full remaining if needed

                    Employee_loan_deduction::create([
                        'employee_loan_id' => $loan->id,
                        'amount'           => $monthlyDeduction,
                        'deduction_date'   => $payPeriod,
                    ]);

                    $loan->decrement('remaining_amount', $monthlyDeduction);
                    $loanDeductions += $monthlyDeduction;
                }
            }

            // 6️⃣ Final salary calculation
            $grossPay = $basicSalary + $totalAllowances;
            $totalDeductions = $otherDeductions + $loanDeductions;
            $netPay = $grossPay - $totalDeductions;

            // 7️⃣ Save Payroll record
            Payroll::create([
                'employee_id'       => $employee->id,
                'pay_period'        => $payPeriod,
                'basic_salary'      => $basicSalary,
                'total_allowances'  => $totalAllowances,
                'total_deductions'  => $otherDeductions, // non-loan deductions only
                'loan_deductions'   => $loanDeductions,
                'gross_pay'         => $grossPay,
                'net_pay'           => $netPay,
            ]);
        }

        // 8️⃣ Update run status
        $run->update(['status' => 'processed']);

        Payroll_run_employee::where('payroll_run_id', $run->id)
            ->update(['status' => 'processed']);

        return back()->with('success', 'Payroll processed successfully');
    }


    public function showMonthPayroll($period = null)
    {
        // Default to current month if not provided
        $period = $period ?? now()->format('Y-m');

        // Make sure it's in the right format
        if (!str_contains($period, '-')) {
            $period .= '-' . now()->format('m'); // append current month
        }

        [$year, $month] = explode('-', $period);

        $payrolls = Payroll::with(['employee.loans.deductions'])
            ->whereYear('pay_period', $year)
            ->whereMonth('pay_period', $month)
            ->get();

        $payrolls->transform(function ($payroll) {
            $totalLoanDeduction = 0;

            if ($payroll->employee) {
                foreach ($payroll->employee->loans as $loan) {
                    foreach ($loan->deductions as $deduction) {
                        $totalLoanDeduction += $deduction->amount;
                    }
                }
            }

            $payroll->total_loan_deductions = $totalLoanDeduction;
            $payroll->net_pay = $payroll->net_pay ??
                ($payroll->basic_salary
                    + $payroll->total_allowances
                    - $payroll->total_deductions
                    - $totalLoanDeduction);

            return $payroll;
        });

        return view('backend.payroll.payroll_monthly', compact('payrolls', 'month', 'year'));
    }

















//    public function generatePayslip($id)
//    {
//        $payroll = Payroll::with(['employee.loans.deductions'])->findOrFail($id);
//
//        // Calculate total loan deductions
//        $totalLoanDeduction = 0;
//        foreach($payroll->employee->loans as $loan) {
//            foreach($loan->deductions as $deduction) {
//                $totalLoanDeduction += $deduction->amount;
//            }
//        }
//
//        // Calculate net pay
//        $netPay = $payroll->basic_salary
//            + $payroll->total_allowances
//            - $payroll->total_deductions
//            - $totalLoanDeduction;
//
//        return Pdf::loadView('backend.payroll.individual_payslip', compact('payroll', 'totalLoanDeduction', 'netPay'))
//            ->stream('Payslip_'.$payroll->employee->full_name.'.pdf');
//    }




    public function generatePayslip($id)
    {
        $payroll = Payroll::with([
            'employee',
            'employee.loans.deductions',
            'employee.deductions',
            'employee.allowances'
        ])->findOrFail($id);

        $year  = date('Y', strtotime($payroll->pay_period));
        $month = date('m', strtotime($payroll->pay_period));

        $loanDeductions = [];
        $otherDeductions = [];
        $totalLoanDeduction = 0;
        $totalOtherDeduction = 0;

        /** ✅ Loan deductions */
        if ($payroll->employee) {
            foreach ($payroll->employee->loans as $loan) {
                foreach ($loan->deductions as $deduction) {
                    if (
                        date('Y-m', strtotime($deduction->deduction_date))
                        ===
                        "$year-$month"
                    ) {
                        $loanDeductions[] = $deduction;
                        $totalLoanDeduction += $deduction->amount;
                    }
                }
            }

            /** ✅ Other deductions */
            foreach ($payroll->employee->deductions as $deduction) {
                if (
                    date('Y-m', strtotime($deduction->deduction_date))
                    ===
                    "$year-$month"
                ) {
                    $otherDeductions[] = $deduction;
                    $totalOtherDeduction += $deduction->amount;
                }
            }
        }

        $totalDeductions = $totalLoanDeduction + $totalOtherDeduction;

        $netPay = $payroll->gross_pay - $totalDeductions;

        return Pdf::loadView(
            'backend.payroll.individual_payslip',
            compact(
                'payroll',
                'loanDeductions',
                'otherDeductions',
                'totalLoanDeduction',
                'totalOtherDeduction',
                'totalDeductions',
                'netPay'
            )
        )->stream(
            'Payslip_' . $payroll->employee->full_name . '.pdf'
        );
    }




    public function showPaidPayrolls()
    {
        // Load all payrolls with employee, loans, and loan deductions
        $payrolls = Payroll::with(['employee.loans.deductions'])
            ->orderBy('pay_period', 'desc')
            ->get();

        // Calculate loan deductions and net pay
        $payrolls->transform(function ($payroll) {
            $totalLoanDeduction = 0;

            if ($payroll->employee) {
                foreach ($payroll->employee->loans as $loan) {
                    foreach ($loan->deductions as $deduction) {
                        $totalLoanDeduction += $deduction->amount;
                    }
                }
            }

            $payroll->total_loan_deductions = $totalLoanDeduction;

            $payroll->net_pay = $payroll->basic_salary + $payroll->total_allowances - $payroll->total_deductions - $totalLoanDeduction;

            return $payroll;
        });

        return view('backend.payroll.paid_payrolls', compact('payrolls'));
    }
//    public function showMonthPayroll($month)
//    {
//        $year = now()->year; // current year
//
//        // Load all employees with their allowances, deductions, loans & loan deductions
//        $employees = Employee::with(['allowances', 'deductions', 'loans.deductions'])->get();
//
//        // Load existing payrolls for the month
//        $payrolls = Payroll::with(['employee.loans.deductions'])
//            ->whereYear('pay_period', $year)
//            ->whereMonth('pay_period', $month)
//            ->get()
//            ->keyBy('employee_id'); // map by employee_id for easy lookup
//
//        return view('backend.payroll.payroll_monthly', compact('employees', 'payrolls', 'month'));
//    }

    // Generate payroll for single employee (called from Blade)
    public function runSinglePayroll(Request $request, $employee_id)
    {
        $year = now()->year;
        $month = now()->month;

        $employee = Employee::with(['allowances','deductions','loans.deductions'])->findOrFail($employee_id);

        // Calculate total allowances
        $totalAllowances = $employee->allowances->sum('amount');

        // Calculate total deductions
        $totalDeductions = $employee->deductions->sum('amount');

        // Calculate total loan deductions
        $totalLoanDeduction = 0;
        foreach($employee->loans as $loan){
            foreach($loan->deductions as $deduction){
                $totalLoanDeduction += $deduction->amount;
            }
        }

        $netPay = $employee->basic_salary + $totalAllowances - $totalDeductions - $totalLoanDeduction;

        // Create payroll record
        $payroll = Payroll::create([
            'employee_id' => $employee->id,
            'pay_period' => now()->format('Y-m-01'),
            'basic_salary' => $employee->basic_salary,
            'total_allowances' => $totalAllowances,
            'total_deductions' => $totalDeductions,
            'loan_deductions' => $totalLoanDeduction,
            'net_pay' => $netPay
        ]);

        return redirect()->back()->with('success', 'Payroll generated for ' . $employee->full_name);
    }

    // Generate individual payslip PDF

    // Generate payroll for an employee and immediately open the payslip
    public function runAndPayslip(Request $request, $employee_id)
    {
        $employee = Employee::with(['allowances','deductions','loans.deductions'])->findOrFail($employee_id);

        // Calculate totals
        $totalAllowances = $employee->allowances->sum('amount');
        $totalDeductions = $employee->deductions->sum('amount');
        $totalLoanDeduction = 0;
        foreach($employee->loans as $loan){
            foreach($loan->deductions as $deduction){
                $totalLoanDeduction += $deduction->amount;
            }
        }

        $netPay = $employee->basic_salary + $totalAllowances - $totalDeductions - $totalLoanDeduction;

        // Create payroll record
        $payroll = Payroll::create([
            'employee_id' => $employee->id,
            'pay_period' => now()->format('Y-m-01'),
            'basic_salary' => $employee->basic_salary,
            'total_allowances' => $totalAllowances,
            'total_deductions' => $totalDeductions,
            'loan_deductions' => $totalLoanDeduction,
            'net_pay' => $netPay
        ]);

        // Redirect to payslip PDF
        return redirect()->route('payroll.payslip', $payroll->id);
    }
    public function monthlyReport($period)
    {
        // Expecting YYYY-MM
        [$year, $month] = explode('-', $period);

        $payrolls = Payroll::with('employee')
            ->whereYear('pay_period', $year)
            ->whereMonth('pay_period', $month)
            ->orderBy('employee_id')
            ->get();

        return view(
            'backend.payroll.reports_monthly',
            compact('payrolls', 'year', 'month')
        );
    }

    public function exportMonthlyPayroll($period)
    {
        // Expecting YYYY-MM
        [$year, $month] = explode('-', $period);

        return Excel::download(
            new MonthlyPayrollExport($year, $month),
            "Monthly_Payroll_{$year}_{$month}.xlsx"
        );
    }
    public function monthly(Request $request)
    {
        $month = $request->month;

        $payrolls = collect();
        $summary = [
            'gross'       => 0,
            'deductions'  => 0,
            'net'         => 0,
        ];

        if ($month) {
            [$year, $monthNum] = explode('-', $month);

            $payrolls = Payroll::with('employee')
                ->whereYear('pay_period', $year)
                ->whereMonth('pay_period', $monthNum)
                ->get();

            foreach ($payrolls as $payroll) {
                $summary['gross'] += $payroll->gross_pay;
                $summary['deductions'] +=
                    ($payroll->total_deductions + $payroll->loan_deductions);
                $summary['net'] += $payroll->net_pay;
            }
        }

        return view('backend.payroll.payroll_monthly_report', compact(
            'payrolls',
            'month',
            'summary'
        ));
    }

    private function calculatePayrollFigures($payroll)
    {
        $employee = $payroll->employee;
        $payDate  = $payroll->pay_period;

        $allowances = 0;
        $deductions = 0;
        $loanDeductions = 0;

        if ($employee) {

            // Allowances
            $allowances = $employee->allowances->sum('amount');

            // Normal deductions
            $deductions = $employee->deductions->sum('amount');

            // 🔥 Loan deductions (MONTH-SPECIFIC)
            foreach ($employee->loans as $loan) {
                $loanDeductions += $loan->deductions
                    ->whereBetween(
                        'deduction_date',
                        [
                            date('Y-m-01', strtotime($payDate)),
                            date('Y-m-t', strtotime($payDate))
                        ]
                    )
                    ->sum('amount');
            }
        }

        $grossPay = $payroll->basic_salary + $allowances;
        $netPay   = $grossPay - $deductions - $loanDeductions;

        return [
            'allowances'      => $allowances,
            'deductions'      => $deductions,
            'loan_deductions' => $loanDeductions,
            'gross_pay'       => $grossPay,
            'net_pay'         => $netPay,
        ];
    }

//    public function showYearlyPayroll($year)
//    {
//        // Load all payrolls for the year with related employee, loans, deductions
//        $payrolls = Payroll::with(['employee.loans.deductions'])
//            ->whereYear('pay_period', $year)
//            ->get();
//
//        // Aggregate totals per employee for the year
//        $yearlyPayroll = $payrolls->groupBy('employee_id')->map(function($employeePayrolls) {
//            $employee = $employeePayrolls->first()->employee;
//
//            $basic = $employeePayrolls->sum('basic_salary');
//            $allowances = $employeePayrolls->sum('total_allowances');
//            $gross = $employeePayrolls->sum('gross_pay');
//            $otherDeductions = $employeePayrolls->sum('total_deductions');
//
//            // Calculate loan deductions from employee_loans -> deductions for this year
//            $loanDeductions = 0;
//            if($employee) {
//                foreach($employee->loans as $loan) {
//                    foreach($loan->deductions as $deduction) {
//                        if(date('Y', strtotime($deduction->deduction_date)) == $employeePayrolls->first()->pay_period->format('Y')) {
//                            $loanDeductions += $deduction->amount;
//                        }
//                    }
//                }
//            }
//
//            $net = $gross - $otherDeductions - $loanDeductions;
//
//            return (object)[
//                'employee' => $employee,
//                'basic' => $basic,
//                'allowances' => $allowances,
//                'gross' => $gross,
//                'other_deductions' => $otherDeductions,
//                'loan_deductions' => $loanDeductions,
//                'net' => $net
//            ];
//        });
//
//        // Totals for footer
//        $summary = [
//            'basic' => $yearlyPayroll->sum(fn($p) => $p->basic),
//            'allowances' => $yearlyPayroll->sum(fn($p) => $p->allowances),
//            'gross' => $yearlyPayroll->sum(fn($p) => $p->gross),
//            'deductions' => $yearlyPayroll->sum(fn($p) => $p->other_deductions),
//            'loan' => $yearlyPayroll->sum(fn($p) => $p->loan_deductions),
//            'net' => $yearlyPayroll->sum(fn($p) => $p->net),
//        ];
//
//        return view('backend.payroll.yearly_report', compact('yearlyPayroll', 'year', 'summary'));
//    }
//    public function showYearlyPayroll(Request $request)
//    {
//        $year = $request->year ?? now()->year;
//
//        $payrolls = collect();
//        $summary = [
//            'gross' => 0,
//            'deductions' => 0,
//            'net' => 0,
//        ];
//
//        $payrolls = Payroll::with([
//            'employee',
//            'employee.loans',
//            'employee.allowances',
//            'employee.deductions'
//        ])
//            ->whereYear('pay_period', $year)
//            ->get();
//
//        $payrolls->transform(function ($payroll) use (&$summary) {
//
//            $basicSalary = $payroll->basic_salary;
//
//            $allowances = $payroll->employee->allowances->sum('amount') ?? 0;
//
//            $loanDeductions = $payroll->employee->loans->sum('monthly_deduction') ?? 0;
//
//            $otherDeductions = $payroll->employee->deductions->sum('amount') ?? 0;
//
//            $grossPay = $basicSalary + $allowances;
//
//            $totalDeductions = $loanDeductions + $otherDeductions;
//
//            $netPay = $grossPay - $totalDeductions;
//
//            // Attach calculated values
//            $payroll->gross_pay = $grossPay;
//            $payroll->loan_deductions = $loanDeductions;
//            $payroll->other_deductions = $otherDeductions;
//            $payroll->total_deductions = $totalDeductions;
//            $payroll->net_pay = $netPay;
//
//            // Year summary
//            $summary['gross'] += $grossPay;
//            $summary['deductions'] += $totalDeductions;
//            $summary['net'] += $netPay;
//
//            return $payroll;
//        });
//
//        return view('backend.payroll.yearly_report', compact(
//            'payrolls',
//            'year',
//            'summary'
//        ));
//    }

    public function yearly(Request $request)
    {
        // Step 1: Get year from request or default to current year
        $year = $request->year ?? now()->year;

        // Step 2: Load all payrolls for the year with relationships
        $payrolls = Payroll::with([
            'employee.allowances',  // all allowances
            'employee.deductions',  // other deductions
            'employee.loans.loanDeductions' // loan deductions
        ])
            ->whereYear('pay_period', $year)
            ->get();

        // Step 3: Transform payrolls to calculate totals
        $summary = [
            'gross' => 0,
            'loan_deductions' => 0,
            'other_deductions' => 0,
            'total_deductions' => 0,
            'net' => 0,
        ];

        $payrolls->transform(function ($payroll) use (&$summary, $year) {

            $employee = $payroll->employee;

            // 1️⃣ Sum allowances for the employee
            $allowances = $employee ? $employee->allowances->sum('amount') : 0;

            // 2️⃣ Sum other deductions (employee_deductions)
            $otherDeductions = $employee ? $employee->deductions
                ->whereYear('deduction_date', $year)
                ->sum('amount') : 0;

            // 3️⃣ Sum loan deductions (from employee loans and employee_loan_deductions)
            $loanDeductions = 0;
            if ($employee && $employee->loans) {
                foreach ($employee->loans as $loan) {
                    if ($loan->loanDeductions) {
                        $loanDeductions += $loan->loanDeductions
                            ->whereYear('deduction_date', $year)
                            ->sum('amount');
                    }
                }
            }

            // 4️⃣ Calculate gross pay and net pay
            $grossPay = $payroll->basic_salary + $allowances;
            $totalDeductions = $otherDeductions + $loanDeductions;
            $netPay = $grossPay - $totalDeductions;

            // 5️⃣ Attach calculated values to payroll object
            $payroll->total_allowances = $allowances;
            $payroll->other_deductions = $otherDeductions;
            $payroll->loan_deductions = $loanDeductions;
            $payroll->gross_pay = $grossPay;
            $payroll->total_deductions = $totalDeductions;
            $payroll->net_pay = $netPay;

            // 6️⃣ Update summary
            $summary['gross'] += $grossPay;
            $summary['loan_deductions'] += $loanDeductions;
            $summary['other_deductions'] += $otherDeductions;
            $summary['total_deductions'] += $totalDeductions;
            $summary['net'] += $netPay;

            return $payroll;
        });

        // Step 4: Return view with payrolls and summary
        return view('backend.payroll.yearly_report', compact('payrolls', 'year', 'summary'));
    }

    public function storeDraft(Request $request)
    {
        $request->validate([
            'month' => 'required|date_format:Y-m',
            'employees' => 'required|array',
        ]);

        $payPeriod = $request->month . '-01';

        if (Payroll_run::where('pay_period', $payPeriod)->exists()) {
            return back()->with('error', 'Payroll already exists for this month');
        }

        $run = null;

        DB::transaction(function () use ($request, $payPeriod, &$run) {
            // 1️⃣ Create draft payroll run
            $run = Payroll_run::create([
                'pay_period' => $payPeriod,
                'status' => 'draft',
            ]);

            // 2️⃣ Assign employees to this payroll run
            foreach ($request->employees as $employeeId) {
                Payroll_run_employee::create([
                    'payroll_run_id' => $run->id,
                    'employee_id' => $employeeId, // ✅ now this will work
                ]);
            }
        });

        return redirect()
            ->route('payroll.review', $run->id)
            ->with('success', 'Payroll draft created');
    }





    public function approve(Request $request, $runId)
    {
        $request->validate([
            'employees' => 'required|array|min:1',
        ], [
            'employees.required' => 'Please select at least one employee.',
        ]);

        $run = Payroll_run::with('employees')->findOrFail($runId);

        // Debug: check if employees exist in run
        if ($run->employees->isEmpty()) {
            return back()->with('error', 'No employees found for this payroll run.');
        }

        // Update selected employees
        $affected = Payroll_run_employee::where('payroll_run_id', $run->id)
            ->whereIn('employee_id', $request->employees)
            ->update(['status' => 'approved']);

        if ($affected === 0) {
            return back()->with('error', 'Selected employees could not be approved.');
        }

        // Update payroll run if at least one employee approved
        $run->status = 'approved';
        $run->save();

        return redirect()
            ->route('backend.payroll.payroll_review', $run->id)
            ->with('success', 'Payroll approved successfully');
    }


    public function PayrollRuns()
    {
        $runs = Payroll_run::with(['employees.employee'])
            ->orderBy('created_at', 'desc')
            ->get();

        return view('backend.payroll.payroll_run_index', compact('runs'));
    }
    public function review(Payroll_run $run)
    {

        $run->load([
            'employees.employee.allowances',
            'employees.employee.deductions',
            'employees.employee.loans'
        ]);

        return view('backend.payroll.payroll_review', compact('run'));
    }
    public function approvePayroll(Payroll_run $run)
    {
        if ($run->status !== 'draft') {
            return back()->with('error', 'Payroll already processed');
        }

        DB::transaction(function () use ($run) {

            $run->load([
                'employees.employee.allowances',
                'employees.employee.deductions',
                'employees.employee.loans'
            ]);

            foreach ($run->employees as $item) {

                $employee = $item->employee;
                $basicSalary = $employee->basic_salary ?? 0;

                /*
                |------------------------------------------------
                | 1️⃣ Allowances
                |------------------------------------------------
                */
                $totalAllowances = $employee->allowances->sum(function ($a) use ($basicSalary) {
                    return $a->type === 'fixed'
                        ? $a->amount
                        : ($basicSalary * $a->amount / 100);
                });

                /*
                |------------------------------------------------
                | 2️⃣ Other Deductions
                |------------------------------------------------
                */
                $otherDeductions = $employee->deductions->sum(function ($d) use ($basicSalary) {
                    return $d->type === 'fixed'
                        ? $d->amount
                        : ($basicSalary * $d->amount / 100);
                });

                /*
                |------------------------------------------------
                | 3️⃣ Loan Deductions (NO MINIMUM / FULL AMOUNT)
                |------------------------------------------------
                */
                $loanDeductions = 0;

                foreach ($employee->loans as $loan) {
                    if ($loan->remaining_amount > 0) {

                        $loanDeductions += $loan->remaining_amount;

                        // clear loan fully
                        $loan->update(['remaining_amount' => 0]);
                    }
                }

                /*
                |------------------------------------------------
                | 4️⃣ Salary Totals
                |------------------------------------------------
                */
                $grossPay = $basicSalary + $totalAllowances;
                $netPay   = $grossPay - ($otherDeductions + $loanDeductions);

                /*
                |------------------------------------------------
                | 5️⃣ Save Payroll Record
                |------------------------------------------------
                */
                Payroll::create([
                    'employee_id'      => $employee->id,
                    'pay_period'       => $run->pay_period,
                    'basic_salary'     => $basicSalary,
                    'total_allowances' => $totalAllowances,
                    'total_deductions' => $otherDeductions,
                    'loan_deductions'  => $loanDeductions,
                    'gross_pay'        => $grossPay,
                    'net_pay'          => $netPay,
                ]);

                /*
                |------------------------------------------------
                | 6️⃣ Mark Employee Payroll as Approved
                |------------------------------------------------
                */
                $item->update(['status' => 'approved']);
            }

            /*
            |------------------------------------------------
            | 7️⃣ Mark Payroll Run as Approved
            |------------------------------------------------
            */
            $run->update(['status' => 'approved']);
        });

        return redirect()
            ->route('backend.payroll.runs', $run->id)
            ->with('success', 'Payroll approved and saved successfully');
    }




    public function printPdf(Payroll_run $run)
    {
        if ($run->status !== 'approved') {
            return back()->with('error', 'Payroll must be approved before printing');
        }

        $payrolls = Payroll::with('employee')
            ->where('pay_period', $run->pay_period)
            ->get();

        if ($payrolls->isEmpty()) {
            return back()->with('error', 'No payroll data found for this period');
        }

        $pdf = Pdf::loadView('backend.payroll.payroll_pdf', [
            'run'      => $run,
            'payrolls' => $payrolls,
        ])->setPaper('A4', 'portrait');

        return $pdf->stream('payroll_'.$run->pay_period.'.pdf');
    }












}
